/***********************************************************************************************//**
 * @file		cangw.cpp
 * @brief		Handling functions for CAN Gateway
 * @date		2014-03-07
 * @copyright	Hatteland Display AS
 **************************************************************************************************/

//==================================================================================================
// Includes
//==================================================================================================
#include <cstring>
#include <pthread.h>
#include <unistd.h>
#include <new>
#include <hdj2534.h>
#include "cangw.hpp"

//==================================================================================================
// Namespaces
//==================================================================================================
using namespace canGw;

//==================================================================================================
// Defines
//==================================================================================================


//==================================================================================================
// Typedefs
//==================================================================================================
/// Definition of structure with CAN Gateway's parameters
struct GwParam_s
{
	/// ID of the device
	unsigned long idDevice;
	/// ID of the CAN CH1
	unsigned long idCh1;
	/// ID of the CAN CH2
	unsigned long idCh2;
	/// ID of the filter for CAN CH1
	unsigned long idFilterCh1;
	/// ID of the filter for CAN CH2
	unsigned long idFilterCh2;
	/// Baud rate of CAN CH1
	unsigned long baudCh1;
	/// Baud rate of CAN CH2
	unsigned long baudCh2;
	/// CAN CH1 active status
	bool activeCh1;
	/// CAN CH2 active status
	bool activeCh2;
};

/// Definition of structure with thread parameters
struct ThreadParam_s
{
	/// Thread's ID
	pthread_t id;
	/// Thread's attributes
	pthread_attr_t attr;
	/// Thread's status
	bool active;
	/// CAN Gateway's channel handled by thread
	Channel_e gwChannel;
};

//==================================================================================================
// Declarations of local functions
//==================================================================================================
static ErrCode_e canGwOpen();
static ErrCode_e canGwClose();
static ErrCode_e canGwConnectChannels();
static ErrCode_e canGwDisconnectChannels();
static ErrCode_e canGwSetFilters();
static ErrCode_e canGwInitRcvThread(Channel_e channel);
static ErrCode_e canGwUninitRcvThread(Channel_e channel);
static void * canGwRcvThreadFunc(void * params);

//==================================================================================================
// Variables
//==================================================================================================
/// Structure with CAN device parameters
static GwParam_s gwParam = {0, 0, 0, 0, 0, 0, 0, false,	false};
/// Structure with thread parameters for Channel1
static ThreadParam_s threadParamCh1;
/// Structure with thread parameters for Channel2
static ThreadParam_s threadParamCh2;
/// Mutex for CAN Gateway's API
static pthread_mutex_t gwMutex = PTHREAD_MUTEX_INITIALIZER;
/// Receive callback function for Channel 1
static RcvCb_t rcvCbCh1;
/// Receive callback function for Channel 2
static RcvCb_t rcvCbCh2;
/// Pointer to receive buffer with PASSTHRU messages for Channel 1
static J2534::PASSTHRU_MSG * rxPassThruMsgCh1 = NULL;
/// Pointer to receive buffer with PASSTHRU messages for Channel 2
static J2534::PASSTHRU_MSG * rxPassThruMsgCh2 = NULL;

/// Maximum number of messages to receive
static const unsigned int MAX_RX_MSGS = 20;
/// Timeout for PassThruWriteMsgs() function
static unsigned long TX_TIMEOUT = 100;
/// Value for baud rate = 125 kbps
static const unsigned long BAUD_VAL_125KBPS = 125000;
/// Value for baud rate = 250 kbps
static const unsigned long BAUD_VAL_250KBPS = 250000;
/// Value for baud rate = 500 kbps
static const unsigned long BAUD_VAL_500KBPS = 500000;
/// Value for baud rate = 1 Mbps
static const unsigned long BAUD_VAL_1MBPS = 1000000;

//==================================================================================================
//==================================================================================================

/***********************************************************************************************//**
 * @brief Function initializes CAN Gateway device
 * @param[in] baudChan1 Baud rate for physical channel 1
 * @param[in] baudChan2 Baud rate for physical channel 2
 * @param[in] cbChan1 Callback function for receive on physical channel 1
 * @param[in] cbChan2 Callback function for receive on physical channel 2
 * @return CAN_GW_OK No error
 * @return CAN_GW_ERR An error occurred
 **************************************************************************************************/
ErrCode_e canGw::init(Baudrate_e baudChan1, Baudrate_e baudChan2, const RcvCb_t cbChan1,
		const RcvCb_t cbChan2)
{
	ErrCode_e err = CAN_GW_OK;

	if ((NULL != cbChan1) && (NULL != cbChan2))
	{
		rcvCbCh1 = cbChan1;
		rcvCbCh2 = cbChan2;
		switch (baudChan1)
		{
			case BAUD_125K:
			default:
				gwParam.baudCh1 = BAUD_VAL_125KBPS;
				break;
			case BAUD_250K:
				gwParam.baudCh1 = BAUD_VAL_250KBPS;
				break;
			case BAUD_500K:
				gwParam.baudCh1 = BAUD_VAL_500KBPS;
				break;
			case BAUD_1M:
				gwParam.baudCh1 = BAUD_VAL_1MBPS;
				break;
		}
		switch (baudChan2)
		{
			case BAUD_125K:
			default:
				gwParam.baudCh2 = BAUD_VAL_125KBPS;
				break;
			case BAUD_250K:
				gwParam.baudCh2 = BAUD_VAL_250KBPS;
				break;
			case BAUD_500K:
				gwParam.baudCh2 = BAUD_VAL_500KBPS;
				break;
			case BAUD_1M:
				gwParam.baudCh2 = BAUD_VAL_1MBPS;
				break;
		}
		if (NULL == rxPassThruMsgCh1)
		{
			rxPassThruMsgCh1 = new (std::nothrow) J2534::PASSTHRU_MSG [MAX_RX_MSGS];
			if (NULL == rxPassThruMsgCh1)
			{
				err = CAN_GW_ERR;
			}
		}
		if (NULL == rxPassThruMsgCh2)
		{
			rxPassThruMsgCh2 = new (std::nothrow) J2534::PASSTHRU_MSG [MAX_RX_MSGS];
			if (NULL == rxPassThruMsgCh2)
			{
				err = CAN_GW_ERR;
			}
		}
		if (CAN_GW_OK == err)
		{
			err = canGwOpen();
			if (CAN_GW_OK == err)
			{
				err = canGwInitRcvThread(CHANNEL_1);
				if (CAN_GW_OK == err)
				{
					err = canGwInitRcvThread(CHANNEL_2);
				}
			}
		}
	}
	return err;
}
/***********************************************************************************************//**
 * @brief Function uninitializes CAN Gateway device
 * @return CAN_GW_OK No error
 * @return CAN_GW_ERR An error occurred
 **************************************************************************************************/
ErrCode_e canGw::uninit()
{
	ErrCode_e err = canGwUninitRcvThread(CHANNEL_1);
	if (CAN_GW_OK == err)
	{
		err = canGwUninitRcvThread(CHANNEL_2);
		if (CAN_GW_OK == err)
		{
			err = canGwClose();
			if (CAN_GW_OK == err)
			{
				delete [] rxPassThruMsgCh1;
				rxPassThruMsgCh1 = NULL;
				delete [] rxPassThruMsgCh2;
				rxPassThruMsgCh2 = NULL;
			}
		}
	}
	return err;
}
/***********************************************************************************************//**
 * @brief Function sends PASSTHRU frame via given channel
 * @param[in] channel Physical channel to be used for sending of a message
 * @param[in] txPassThruMsg Pointer to a message to be sent
 * @return CAN_GW_OK No error
 * @return CAN_GW_ERR An error occurred
 **************************************************************************************************/
ErrCode_e canGw::send(Channel_e channel, J2534::PASSTHRU_MSG * txPassThruMsg)
{
	using namespace J2534;
	unsigned long msgNum = 1;
	unsigned long idCh;
	ErrCode_e err = CAN_GW_ERR;
	J2534_ERROR_CODE passThruErr = STATUS_NOERROR;

	switch (channel)
	{
		case CHANNEL_1:
		default:
			idCh = gwParam.idCh1;
			break;
		case CHANNEL_2:
			idCh = gwParam.idCh2;
			break;
	}
	pthread_mutex_lock(&gwMutex);
	passThruErr = PassThruWriteMsgs(idCh, txPassThruMsg, &msgNum, TX_TIMEOUT);
	pthread_mutex_unlock(&gwMutex);
	if (STATUS_NOERROR == passThruErr)
	{
		err = CAN_GW_OK;
	}
	return err;
}
/***********************************************************************************************//**
 * @brief Function starts CAN Gateway's periodic message
 * @param[in] channel Physical channel to be used for sending of a periodic message
 * @param[in] txPassThruMsg Pointer to a message to be sent periodically
 * @param[in] interval Interval for a periodic message in miliseconds
 * @return CAN_GW_OK No error
 * @return CAN_GW_ERR An error occurred
 **************************************************************************************************/
ErrCode_e canGw::startPeriodic(Channel_e channel, J2534::PASSTHRU_MSG * txPassThruMsg,
		unsigned long interval)
{
	using namespace J2534;
	// Note that below idMsg handle is an automatic variable allocated on stack so in this
	// implementation periodic message will not be terminated with PassThruStopPeriodicMsg()
	// because idMsg will be forgotten. Started periodic messages will be disabled at channel
	// disconnection automatically by CAN Gateway.
	unsigned long idMsg;
	unsigned long idCh;
	ErrCode_e err = CAN_GW_ERR;
	J2534_ERROR_CODE passThruErr = STATUS_NOERROR;

	switch (channel)
	{
		case CHANNEL_1:
		default:
			idCh = gwParam.idCh1;
			break;
		case CHANNEL_2:
			idCh = gwParam.idCh2;
			break;
	}
	pthread_mutex_lock(&gwMutex);
	passThruErr = PassThruStartPeriodicMsg(idCh, txPassThruMsg, &idMsg, interval);
	pthread_mutex_unlock(&gwMutex);
	if (STATUS_NOERROR == passThruErr)
	{
		err = CAN_GW_OK;
	}
	return err;
}

//==================================================================================================
//==================================================================================================

/***********************************************************************************************//**
 * @brief Function opens CAN Gateway
 * @return CAN_GW_OK No error
 * @return CAN_GW_ERR An error occurred
 **************************************************************************************************/
static ErrCode_e canGwOpen()
{
	using namespace J2534;
	ErrCode_e err = CAN_GW_ERR;

	J2534_ERROR_CODE passThruErr = PassThruOpen(NULL, &gwParam.idDevice);
	if ( STATUS_NOERROR ==  passThruErr)
	{
		err = canGwConnectChannels();
		if (CAN_GW_OK == err)
		{
			err = canGwSetFilters();
		}
	}
	return err;
}
/***********************************************************************************************//**
 * @brief Function closes CAN Gateway
 * @return CAN_GW_OK No error
 * @return CAN_GW_ERR An error occurred
 **************************************************************************************************/
static ErrCode_e canGwClose()
{
	using namespace J2534;

	ErrCode_e err = canGwDisconnectChannels();
	if (CAN_GW_OK == err)
	{
		J2534_ERROR_CODE passThruErr = PassThruClose(gwParam.idDevice);
		if (STATUS_NOERROR != passThruErr)
		{
			err = CAN_GW_ERR;
		}
	}
	return err;
}
/***********************************************************************************************//**
 * @brief Function connects CAN Gateway's channels
 * @return CAN_GW_OK No error
 * @return CAN_GW_ERR An error occurred
 **************************************************************************************************/
static ErrCode_e canGwConnectChannels()
{
	using namespace J2534;
	ErrCode_e err = CAN_GW_OK;

	if (false == gwParam.activeCh1)
	{
		static const unsigned long flagsCh1 = 0;
		J2534_ERROR_CODE passThruErr = PassThruConnect(gwParam.idDevice, CAN, flagsCh1,
				gwParam.baudCh1, &gwParam.idCh1);
		if (STATUS_NOERROR == passThruErr)
		{
			gwParam.activeCh1 = true;
		}
		else
		{
			err = CAN_GW_ERR;
		}
	}
	if (CAN_GW_OK == err)
	{
		if (false == gwParam.activeCh2)
		{
			static const unsigned long flagsCh2 = PHYSICAL_CHANNEL;
			J2534_ERROR_CODE passThruErr = PassThruConnect(gwParam.idDevice, CAN, flagsCh2,
					gwParam.baudCh2, &gwParam.idCh2);
			if (STATUS_NOERROR == passThruErr)
			{
				gwParam.activeCh2 = true;
			}
			else
			{
				err = CAN_GW_ERR;
			}
		}
	}

	return err;
}
/***********************************************************************************************//**
 * @brief Function disconnects CAN Gateway's channels
 * @return CAN_GW_OK No error
 * @return CAN_GW_ERR An error occurred
 **************************************************************************************************/
static ErrCode_e canGwDisconnectChannels()
{
	using namespace J2534;
	ErrCode_e err = CAN_GW_OK;

	if (false != gwParam.activeCh1)
	{
		J2534_ERROR_CODE passThruErr = PassThruDisconnect(gwParam.idCh1);
		if (STATUS_NOERROR == passThruErr)
		{
			gwParam.activeCh1 = false;
		}
		else
		{
			err = CAN_GW_ERR;
		}
	}
	if (CAN_GW_OK == err)
	{
		if (false != gwParam.activeCh2)
		{
			J2534_ERROR_CODE passThruErr = PassThruDisconnect(gwParam.idCh2);
			if (STATUS_NOERROR == passThruErr)
			{
				gwParam.activeCh2 = false;
			}
			else
			{
				err = CAN_GW_ERR;
			}
		}
	}
	return err;
}
/***********************************************************************************************//**
 * @brief Function sets CAN Gateway's message filters
 * @return CAN_GW_OK No error
 * @return CAN_GW_ERR An error occurred
 **************************************************************************************************/
static ErrCode_e canGwSetFilters()
{
	using namespace J2534;
	ErrCode_e err = CAN_GW_OK;

	J2534_ERROR_CODE passThruErr = PassThruIoctl(gwParam.idCh1, CLEAR_MSG_FILTERS, NULL, NULL);
	if (STATUS_NOERROR != passThruErr)
	{
		err = CAN_GW_ERR;
	}
	passThruErr = PassThruIoctl(gwParam.idCh2, CLEAR_MSG_FILTERS, NULL, NULL);
	if (STATUS_NOERROR != passThruErr)
	{
		err = CAN_GW_ERR;
	}
	if (CAN_GW_OK == err)
	{
		PASSTHRU_MSG maskPassThruMsg;
		PASSTHRU_MSG patternPassThruMsg;
		J2534_ConnectFlags filterFlags;

		memset(&maskPassThruMsg, 0, sizeof(PASSTHRU_MSG));
		maskPassThruMsg.ProtocolID = CAN;
		filterFlags.bits.Can29BitId = 0;
		maskPassThruMsg.TxFlags = filterFlags.value;
		memset(&patternPassThruMsg, 0, sizeof(PASSTHRU_MSG));
		patternPassThruMsg.ProtocolID = CAN;
		filterFlags.bits.Can29BitId = 0;
		patternPassThruMsg.TxFlags = filterFlags.value;

		passThruErr = PassThruStartMsgFilter(gwParam.idCh1, PASS_FILTER, &maskPassThruMsg,
				&patternPassThruMsg, NULL, &gwParam.idFilterCh1);
		if (STATUS_NOERROR != passThruErr)
		{
			err = CAN_GW_ERR;
		}
		passThruErr = PassThruStartMsgFilter(gwParam.idCh2, PASS_FILTER, &maskPassThruMsg,
				&patternPassThruMsg, NULL, &gwParam.idFilterCh2);
		if (STATUS_NOERROR != passThruErr)
		{
			err = CAN_GW_ERR;
		}
	}
	return err;
}
/***********************************************************************************************//**
 * @brief Function initializes receiving thread for given physical channel
 * @param[in] channel CAN Gateway's physical channel
 * @return CAN_GW_OK No error
 * @return CAN_GW_ERR An error occurred
 **************************************************************************************************/
static ErrCode_e canGwInitRcvThread(Channel_e channel)
{
	ErrCode_e err = CAN_GW_OK;
	ThreadParam_s * threadParam;

	switch (channel)
	{
		case CHANNEL_1:
		default:
			threadParam = &threadParamCh1;
			break;
		case CHANNEL_2:
			threadParam = &threadParamCh2;
			break;
	}
	threadParam->gwChannel = channel;
	threadParam->active = true;
	threadParam->id = 0;
	int retVal = pthread_attr_init(&threadParam->attr);
	if ( 0 == retVal )
	{
		retVal = pthread_attr_setdetachstate(&threadParam->attr, PTHREAD_CREATE_JOINABLE);
		if ( 0 == retVal )
		{
			retVal = pthread_create(&threadParam->id, &threadParamCh1.attr, &canGwRcvThreadFunc,
					threadParam);
			if ( 0 == retVal )
			{
				pthread_attr_destroy(&threadParam->attr);
			}
			else
			{
				err = CAN_GW_ERR;
				threadParam->active = false;
			}
		}
		else
		{
			threadParam->active = false;
			err = CAN_GW_ERR;
		}
	}
	else
	{
		err = CAN_GW_ERR;
	}
	return err;
}
/***********************************************************************************************//**
 * @brief Function uninitializes receiving thread for given physical channel
 * @param[in] channel CAN Gateway's physical channel
 * @return CAN_GW_OK No error
 * @return CAN_GW_ERR An error occurred
 **************************************************************************************************/
static ErrCode_e canGwUninitRcvThread(Channel_e channel)
{
	ErrCode_e err = CAN_GW_OK;
	ThreadParam_s * threadParam;

	switch (channel)
	{
		case CHANNEL_1:
		default:
			threadParam = &threadParamCh1;
			break;
		case CHANNEL_2:
			threadParam = &threadParamCh2;
			break;
	}
	if (false != threadParam->active)
	{
		int retVal = pthread_cancel(threadParam->id);
		if (0 == retVal)
		{
			pthread_join(threadParam->id, NULL);
			threadParam->active = false;
		}
		else
		{
			err = CAN_GW_ERR;
		}
	}
	return err;
}
/***********************************************************************************************//**
 * @brief Function handles CAN Gateway's receiving threads
 * @param[in] params Pointer to structure {@link ThreadParam_s} with thread parameters
 * @return Pointer to status of the thread - always NULL
 **************************************************************************************************/
static void * canGwRcvThreadFunc(void * params)
{
	using namespace J2534;
	ThreadParam_s * threadParam = reinterpret_cast<ThreadParam_s *>(params);
	unsigned long msgNum;
	// In this example timeout is not used
	static unsigned long RX_TIMEOUT = 0;
	// 25 ms delay for thread loops
	static const unsigned int THREAD_DELAY = 25 * 1000;

	if (CHANNEL_1 == threadParam->gwChannel)
	{
		while (1)
		{
			msgNum = MAX_RX_MSGS;
			pthread_mutex_lock(&gwMutex);
			J2534_ERROR_CODE passThruErr = PassThruReadMsgs(gwParam.idCh1, rxPassThruMsgCh1,
					&msgNum, RX_TIMEOUT);
			pthread_mutex_unlock(&gwMutex);
			if (STATUS_NOERROR == passThruErr)
			{
				rcvCbCh1(rxPassThruMsgCh1, msgNum);
			}
			else if (ERR_BUFFER_EMPTY == passThruErr)
			{
				; // User defined action
			}
			else
			{
				; // User defined error handling
			}
			usleep(THREAD_DELAY);
		}
	}
	else if (CHANNEL_2 == threadParam->gwChannel)
	{
		while (1)
		{
			msgNum = MAX_RX_MSGS;
			pthread_mutex_lock(&gwMutex);
			J2534_ERROR_CODE passThruErr = PassThruReadMsgs(gwParam.idCh2, rxPassThruMsgCh2,
					&msgNum, RX_TIMEOUT);
			pthread_mutex_unlock(&gwMutex);
			if (STATUS_NOERROR == passThruErr)
			{
				rcvCbCh2(rxPassThruMsgCh2, msgNum);
			}
			else if (ERR_BUFFER_EMPTY == passThruErr)
			{
				; // User defined action
			}
			else
			{
				; // User defined error handling
			}
			usleep(THREAD_DELAY);
		}
	}
	else
	{
		; // never used
	}
	return NULL;
}
/**************************************************************************************************/
